<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\User\Block;

use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Tables\User\UserBlockTable;
use Erlage\Photogram\Data\Models\User\UserModelHelper;
use Erlage\Photogram\Data\Models\User\Block\UserBlockFinder;
use Erlage\Photogram\Data\Models\User\Block\UserBlockBuilder;
use Erlage\Photogram\Data\Models\User\Follow\UserFollowFinder;

final class UserBlockActions extends ExceptionalRequests
{
    /*
    |--------------------------------------------------------------------------
    | follow user
    |--------------------------------------------------------------------------
    */

    public static function add(): void
    {
        self::blockProcessor(true);
    }

    /*
    |--------------------------------------------------------------------------
    | unfollow user
    |--------------------------------------------------------------------------
    */

    public static function remove(): void
    {
        self::blockProcessor(false);
    }

    /*
    |--------------------------------------------------------------------------
    | follow/unfollow user
    |--------------------------------------------------------------------------
    */

    private static function blockProcessor(bool $doBlock): void
    {
        self::process(function () use ($doBlock)
        {
            /*
            |--------------------------------------------------------------------------
            | flags
            |--------------------------------------------------------------------------
            */

            $flagUpdateUserBlockCache = false;

            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $userIdFromReq = self::$request -> findKey(UserTable::ID, RequestTable::PAYLOAD, UserTable::TABLE_NAME);

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $userIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | ensure target user exists
            |--------------------------------------------------------------------------
            */

            $targetUserModel = UserModel::findFromId_throwException($userIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | try to get block model, this tells whether target user is blocked or not
            |--------------------------------------------------------------------------
            */

            $userBlockFinder = (new UserBlockFinder())
                -> setBlockedUserId($targetUserModel -> getId())
                -> setBlockedByUserId(self::$authedUserModel -> getId())
                -> find();

            /*
            |--------------------------------------------------------------------------
            | if request is to un block the user
            |--------------------------------------------------------------------------
            */

            if (false == $doBlock)
            {
                if ($userBlockFinder -> isFound())
                {
                    $userBlockFinder -> popModelFromResults() -> delete();

                    $flagUpdateUserBlockCache = true;
                }
            }

            /*
            |--------------------------------------------------------------------------
            | else if it's a do block request
            |--------------------------------------------------------------------------
            */
            elseif (true == $doBlock)
            {
                /*
                |--------------------------------------------------------------------------
                | if already blocked
                |--------------------------------------------------------------------------
                */
                if ($userBlockFinder -> isFound())
                {
                    /*
                    |--------------------------------------------------------------------------
                    | add to model to response  map
                    |--------------------------------------------------------------------------
                    */

                    self::addToResponse(
                        UserBlockTable::getTableName(),
                        $userBlockFinder
                            -> popModelFromResults()
                            -> getDataMap()
                    );
                }

                /*
                |--------------------------------------------------------------------------
                | else if user is not already blocked
                |--------------------------------------------------------------------------
                */

                else
                {
                    /*
                    |--------------------------------------------------------------------------
                    | build model
                    |--------------------------------------------------------------------------
                    */

                    $userBlockModel = (new UserBlockBuilder())
                        -> setBlockedUserId($targetUserModel -> getId())
                        -> setBlockedByUserId(self::$authedUserModel -> getId())
                        -> dispense();

                    /*
                    |--------------------------------------------------------------------------
                    | do block
                    |--------------------------------------------------------------------------
                    */

                    $userBlockModel -> save();

                    $flagUpdateUserBlockCache = true;

                    /**
                     * remove target user from current user's lists
                     * this is not required, but it makes sense to invalidate dependencies
                     */

                    /*
                    |--------------------------------------------------------------------------
                    | remove target user from followers
                    |--------------------------------------------------------------------------
                    */

                    $userFollowFinder = (new UserFollowFinder())
                        -> setFollowedByUserId($targetUserModel -> getId())
                        -> setFollowedUserId(self::$authedUserModel -> getId())
                        -> find();

                    if ($userFollowFinder -> isFound())
                    {
                        $userFollowFinder -> popModelFromResults() -> delete();

                        // update cache

                        UserModelHelper::updateCacheFollowingsCount($targetUserModel);
                        UserModelHelper::updateCacheFollowersCount(self::$authedUserModel);
                    }

                    /*
                    |--------------------------------------------------------------------------
                    | remove target user from current's user followings
                    | i.e make sure current user isn't following the target user anymore
                    |--------------------------------------------------------------------------
                    */

                    $userFollowFinder -> clean();

                    $userFollowFinder
                        -> setFollowedUserId($targetUserModel -> getId())
                        -> setFollowedByUserId(self::$authedUserModel -> getId())
                        -> find();

                    if ($userFollowFinder -> isFound())
                    {
                        $userFollowFinder -> popModelFromResults() -> delete();

                        // update cache

                        UserModelHelper::updateCacheFollowersCount($targetUserModel);
                        UserModelHelper::updateCacheFollowingsCount(self::$authedUserModel);
                    }

                    /*
                    |--------------------------------------------------------------------------
                    |  if everything is alright, add model to response
                    |--------------------------------------------------------------------------
                    */

                    self::addToResponse(UserBlockTable::getTableName(), $userBlockModel -> getDataMap());

                    /*
                    |--------------------------------------------------------------------------
                    |  add target user to response
                    |--------------------------------------------------------------------------
                    */

                    self::addToResponse(UserTable::getTableName(), $targetUserModel -> getDataMap());
                }
            }

            /*
            |--------------------------------------------------------------------------
            | update current user's blocked cache
            |--------------------------------------------------------------------------
            */

            if ($flagUpdateUserBlockCache)
            {
                $blockedUserIdsDTO = self::$authedUserModel -> getCacheBlockedUserIds();

                if (false == $doBlock)
                {
                    $blockedUserIdsDTO -> removeValue($targetUserModel -> getId());
                }
                elseif ( ! $blockedUserIdsDTO -> containsValue($targetUserModel -> getId()))
                {
                    $blockedUserIdsDTO -> push($targetUserModel -> getId());
                }

                self::$authedUserModel -> update(
                    array(
                        UserTable::CACHE_BLOCKED_USER_IDS => $blockedUserIdsDTO,
                    )
                );

                self::$authedUserModel -> save();
            }
        });
    }
}
